import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Film, Calendar, CheckCircle2, ArrowRight, Monitor, Video, Scissors, Play, Award, Users, Clock, Zap } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedSoftware, setSelectedSoftware] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const softwareOptions = [
    { 
      id: 'premiere', 
      name: 'Premiere Pro', 
      icon: Monitor,
      gradient: 'linear-gradient(135deg, #E50914 0%, #FF6B35 100%)',
      description: 'Профессиональный монтаж для кино и ТВ',
      features: ['Цветокоррекция', 'Аудио', 'Эффекты', 'Экспорт']
    },
    { 
      id: 'davinci', 
      name: 'DaVinci Resolve', 
      icon: Video,
      gradient: 'linear-gradient(135deg, #FF6B35 0%, #FFD23F 100%)',
      description: 'Мощная система для цветокоррекции',
      features: ['Fusion', 'Fairlight', 'Color', 'Deliver']
    },
    { 
      id: 'finalcut', 
      name: 'Final Cut', 
      icon: Scissors,
      gradient: 'linear-gradient(135deg, #FFD23F 0%, #E50914 100%)',
      description: 'Быстрый монтаж для Mac',
      features: ['Magnetic Timeline', 'Multicam', 'HDR', '360°']
    },
    { 
      id: 'capcut', 
      name: 'CapCut', 
      icon: Play,
      gradient: 'linear-gradient(135deg, #E50914 0%, #FFD23F 100%)',
      description: 'Простой монтаж для соцсетей',
      features: ['Шаблоны', 'Эффекты', 'Музыка', 'Текст']
    }
  ]

  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (!selectedSoftware || !selectedDate) return

    const saved = localStorage.getItem('videoProjects')
    const projects = saved ? JSON.parse(saved) : []
    
    const softwareData = softwareOptions.find(s => s.id === selectedSoftware)
    
    const newProject = {
      id: Date.now(),
      course: `${softwareData.name}: Базовый курс`,
      instructor: 'Преподаватель будет назначен',
      work: 'Первый проект',
      rating: null,
      photo: '/images/project-placeholder.jpg'
    }
    
    projects.push(newProject)
    localStorage.setItem('videoProjects', JSON.stringify(projects))
    
    setIsSubmitted(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="enroll-film-page">
      <div className="film-strip-top"></div>
      
      <div className="container">
        <motion.section 
          className="enroll-header-film"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="enroll-badge-film"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ delay: 0.2, type: "spring" }}
          >
            <Film size={18} strokeWidth={2.5} />
            <span>НОВАЯ ЗАПИСЬ</span>
          </motion.div>
          <h1 className="enroll-title-film">
            Записаться на <span className="title-accent-film">курс</span>
          </h1>
          <p className="enroll-description-film">
            Выберите программное обеспечение и дату старта. Начните свой путь в видеомонтаже!
          </p>
        </motion.section>

        <div className="enroll-layout-film">
          <motion.section 
            className="form-section-film"
            initial={{ opacity: 0, x: -30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
          >
            <form className="form-film" onSubmit={handleSubmit}>
              <div className="form-group-film">
                <label className="form-label-film">
                  <Monitor size={24} strokeWidth={2} />
                  <span>Выберите ПО</span>
                </label>
                <div className="software-grid-film">
                  {softwareOptions.map((software, index) => {
                    const Icon = software.icon
                    return (
                      <motion.button
                        key={software.id}
                        type="button"
                        className={`software-card-film ${selectedSoftware === software.id ? 'selected' : ''}`}
                        onClick={() => setSelectedSoftware(software.id)}
                        initial={{ opacity: 0, y: 30 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: 0.4 + index * 0.1 }}
                        whileHover={{ scale: 1.02, y: -5 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedSoftware === software.id ? 'var(--film-red)' : 'var(--film-border)',
                          background: selectedSoftware === software.id 
                            ? software.gradient 
                            : 'var(--film-gray)'
                        }}
                      >
                        <div className="software-icon-film">
                          <Icon size={40} strokeWidth={2} />
                        </div>
                        <div className="software-content-film">
                          <h3 className="software-name-film">{software.name}</h3>
                          <p className="software-desc-film">{software.description}</p>
                          <div className="software-features-film">
                            {software.features.map((feature, idx) => (
                              <span key={idx} className="feature-tag-film">{feature}</span>
                            ))}
                          </div>
                        </div>
                        {selectedSoftware === software.id && (
                          <motion.div 
                            className="check-mark-film"
                            initial={{ scale: 0, rotate: -180 }}
                            animate={{ scale: 1, rotate: 0 }}
                            transition={{ type: "spring" }}
                          >
                            <CheckCircle2 size={32} strokeWidth={2.5} fill="currentColor" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <AnimatePresence>
                {selectedSoftware && (
                  <motion.div
                    className="form-group-film date-group-film"
                    initial={{ opacity: 0, height: 0 }}
                    animate={{ opacity: 1, height: 'auto' }}
                    exit={{ opacity: 0, height: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <label className="form-label-film">
                      <Calendar size={24} strokeWidth={2} />
                      <span>Дата старта</span>
                    </label>
                    <div className="date-input-wrapper-film">
                      <Calendar size={20} strokeWidth={2} className="date-icon-film" />
                      <input
                        type="date"
                        value={selectedDate}
                        onChange={(e) => setSelectedDate(e.target.value)}
                        min={minDateStr}
                        max={maxDateStr}
                        className="date-input-film"
                        required
                      />
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>

              <motion.button 
                type="submit" 
                className="submit-btn-film"
                disabled={!selectedSoftware || !selectedDate}
                whileHover={{ scale: selectedSoftware && selectedDate ? 1.05 : 1 }}
                whileTap={{ scale: selectedSoftware && selectedDate ? 0.95 : 1 }}
              >
                <span>Записаться</span>
                <ArrowRight size={22} strokeWidth={2.5} />
              </motion.button>
            </form>

            {isSubmitted && (
              <motion.div 
                className="success-film"
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
              >
                <CheckCircle2 size={40} strokeWidth={2} fill="currentColor" />
                <span>Запись оформлена! Переходим к вашим проектам...</span>
              </motion.div>
            )}
          </motion.section>

          <aside className="info-panel-film">
            <motion.div 
              className="info-card-film"
              initial={{ opacity: 0, x: 30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.5 }}
            >
              <div className="info-header-film">
                <Award size={28} strokeWidth={2} />
                <h3 className="info-title-film">Преимущества</h3>
              </div>
              <ul className="benefits-list-film">
                <li>
                  <Users size={20} strokeWidth={2} />
                  <span>Опытные преподаватели</span>
                </li>
                <li>
                  <Clock size={20} strokeWidth={2} />
                  <span>Гибкий график</span>
                </li>
                <li>
                  <Zap size={20} strokeWidth={2} />
                  <span>Практические проекты</span>
                </li>
                <li>
                  <Award size={20} strokeWidth={2} />
                  <span>Сертификат по окончании</span>
                </li>
              </ul>
            </motion.div>

            <motion.div 
              className="info-card-film"
              initial={{ opacity: 0, x: 30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.6 }}
            >
              <div className="info-header-film">
                <Video size={28} strokeWidth={2} />
                <h3 className="info-title-film">Что включено</h3>
              </div>
              <div className="features-list-film">
                <div className="feature-number-item-film">
                  <div className="feature-number-film">01</div>
                  <p>Теоретические основы</p>
                </div>
                <div className="feature-number-item-film">
                  <div className="feature-number-film">02</div>
                  <p>Практические задания</p>
                </div>
                <div className="feature-number-item-film">
                  <div className="feature-number-film">03</div>
                  <p>Разбор ваших работ</p>
                </div>
                <div className="feature-number-item-film">
                  <div className="feature-number-film">04</div>
                  <p>Портфолио проектов</p>
                </div>
              </div>
            </motion.div>
          </aside>
        </div>

        <ImageSlider />

        <section className="testimonials-film-section">
          <motion.h2 
            className="section-title-film"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
          >
            <Film size={32} strokeWidth={2} />
            Отзывы студентов
          </motion.h2>
          <div className="testimonials-grid-film">
            {[
              { text: 'Отличный курс по Premiere Pro! Научился профессиональному монтажу за короткое время.', author: 'Иван П.' },
              { text: 'DaVinci Resolve открыл для меня новые возможности в цветокоррекции. Рекомендую!', author: 'Елена К.' },
              { text: 'Преподаватели очень внимательные, всегда помогают с вопросами. Спасибо!', author: 'Александр М.' }
            ].map((testimonial, index) => (
              <motion.div 
                key={index}
                className="testimonial-card-film"
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
                whileHover={{ y: -10 }}
              >
                <div className="testimonial-quote-film">"</div>
                <p className="testimonial-text-film">{testimonial.text}</p>
                <div className="testimonial-author-film">— {testimonial.author}</div>
              </motion.div>
            ))}
          </div>
        </section>
      </div>
      
      <div className="film-strip-bottom"></div>
    </div>
  )
}

export default Enroll

